# 멀티 에이전트 Playwright 테스트 자동화

> 범용 웹 애플리케이션 테스트 프레임워크

---

## 다른 프로젝트에서 사용하기

### 1단계: 파일 복사
```bash
# 새 프로젝트로 tests 폴더 전체 복사
cp -r tests/ /path/to/your-project/tests/
```

### 2단계: 의존성 설치
```bash
cd /path/to/your-project

# Playwright 및 테스트 도구 설치
npm install --save-dev playwright @playwright/test axe-core

# Chromium 브라우저 설치
npx playwright install chromium
```

### 3단계: package.json에 스크립트 추가
```json
{
  "scripts": {
    "test": "node tests/coordinator.js",
    "test:headless": "HEADLESS=true node tests/coordinator.js",
    "test:debug": "HEADLESS=false SLOW_MO=500 DEVTOOLS=true node tests/coordinator.js",
    "test:clean": "rm -rf tests/results tests/screenshots tests/logs"
  }
}
```

### 4단계: 포트 설정
```bash
# 방법 1: 환경 변수로 지정
TEST_URL=http://localhost:3000 npm test

# 방법 2: config.js 직접 수정
# tests/shared/config.js의 baseUrl 변경
```

---

## 빠른 시작

### 1. 서버 실행 (터미널 1)
```bash
npm run dev
# 또는 앱이 실행되는 명령어
```

### 2. 테스트 실행 (터미널 2)
```bash
# 기본 실행 (headless)
npm test

# 다른 포트로 실행
TEST_URL=http://localhost:3000 npm test
TEST_URL=http://localhost:8080 npm test

# 브라우저 창 보이면서 실행
npm run test:debug

# 결과 정리
npm run test:clean
```

---

## 아키텍처: 5개 전문 에이전트 병렬 실행

```
┌─────────────────────────────────────────────────────────────┐
│                    Test Coordinator                          │
│         (tests/coordinator.js)                               │
├─────────────────────────────────────────────────────────────┤
│                                                              │
│  Phase 1: Explorer 먼저 실행 (페이지 맵 생성)                   │
│                                                              │
│  Phase 2: 나머지 4개 에이전트 병렬 실행                         │
│           ┌──────────┬──────────┬──────────┬──────────┐      │
│           │Functional│  Visual  │   API    │Performance│      │
│           └──────────┴──────────┴──────────┴──────────┘      │
│                                                              │
│  Phase 3: 결과 집계 및 통합 리포트 생성                         │
│                                                              │
└─────────────────────────────────────────────────────────────┘
```

---

## 환경 변수

| 변수 | 기본값 | 설명 |
|------|--------|------|
| `TEST_URL` | `http://localhost:9005` | 테스트 대상 URL |
| `HEADLESS` | `true` | 브라우저 숨김 여부 |
| `SLOW_MO` | `0` | 동작 지연 시간 (ms) |
| `DEVTOOLS` | `false` | 개발자 도구 열기 |

**사용 예시:**
```bash
# 포트 3000에서 실행, 브라우저 보이게, 느리게
TEST_URL=http://localhost:3000 HEADLESS=false SLOW_MO=200 npm test

# 외부 서버 테스트
TEST_URL=https://staging.example.com npm test
```

---

## 에이전트 상세

### 🔍 Agent 1: Explorer (탐색 전문)
**파일**: `tests/agents/explorer/index.js`

**역할**:
- 모든 페이지/라우트 BFS 크롤링
- DOM 구조 분석 및 셀렉터 추출
- 인터랙티브 요소 카탈로그 생성
- 버튼, 폼, 입력 필드 자동 감지

**출력**: `tests/results/page-map.json`

**커스터마이징**: 특정 경로 제외하려면 `explorePage()` 메서드에 필터 추가

---

### ✅ Agent 2: Functional (기능 테스트 전문)
**파일**: `tests/agents/functional/index.js`

**역할**:
- 페이지 로드 테스트
- 버튼 클릭 가능 여부 테스트
- 폼 유효성 검사
- 탭/네비게이션 테스트

**출력**: `tests/results/functional-tests.json`

**커스터마이징**: 앱 특화 테스트 추가하려면 `run()` 메서드에 새 테스트 함수 호출 추가
```javascript
// 예: 로그인 테스트 추가
await this.testLogin(page);
```

---

### 🎨 Agent 3: Visual (UI/UX 전문)
**파일**: `tests/agents/visual/index.js`

**역할**:
- 반응형 스크린샷 캡처 (모바일/태블릿/데스크톱)
- 레이아웃 오버플로우 검사
- 작은 터치 타겟 감지
- 접근성(a11y) axe-core 검사

**출력**:
- `tests/results/visual-tests.json`
- `tests/screenshots/*.png`

**커스터마이징**: 뷰포트 추가하려면 `config.js`의 `viewports` 수정

---

### 🌐 Agent 4: API (네트워크 전문)
**파일**: `tests/agents/api/index.js`

**역할**:
- 모든 XHR/Fetch 요청 자동 캡처
- API 응답 상태 코드 검증
- 요청 실패 감지
- 에러 응답 로깅

**출력**:
- `tests/results/api-tests.json`
- `tests/results/network-logs.json`

**커스터마이징**: 테스트할 API 엔드포인트 추가하려면 `testApiEndpoints()`의 `commonEndpoints` 배열 수정

---

### ⚡ Agent 5: Performance (성능 전문)
**파일**: `tests/agents/performance/index.js`

**역할**:
- 페이지 로드 시간 측정
- Core Web Vitals 수집 (FCP, LCP, CLS)
- 리소스 타이밍 분석
- 성능 등급 판정 (A/B/C)

**출력**: `tests/results/performance-report.json`

**성능 등급 기준**:
- **A (Good)**: FCP ≤ 1.8s, Load ≤ 3s
- **B (Needs Improvement)**: FCP ≤ 3s, Load ≤ 5s
- **C (Poor)**: 그 외

---

## 설정 파일

**파일**: `tests/shared/config.js`

```javascript
module.exports = {
  // 테스트 대상 URL (환경 변수로 오버라이드 가능)
  baseUrl: process.env.TEST_URL || 'http://localhost:9005',

  // 에이전트별 타임아웃 (ms)
  agents: {
    explorer: { enabled: true, timeout: 15 * 60 * 1000 },
    functional: { enabled: true, timeout: 15 * 60 * 1000 },
    visual: { enabled: true, timeout: 15 * 60 * 1000 },
    api: { enabled: true, timeout: 15 * 60 * 1000 },
    performance: { enabled: true, timeout: 15 * 60 * 1000 }
  },

  // 브라우저 설정
  browser: {
    headless: process.env.HEADLESS !== 'false',
    slowMo: parseInt(process.env.SLOW_MO) || 0,
    devtools: process.env.DEVTOOLS === 'true'
  },

  // 반응형 테스트 뷰포트
  viewports: {
    mobile: { width: 375, height: 667 },
    tablet: { width: 768, height: 1024 },
    desktop: { width: 1920, height: 1080 }
  },

  // 결과 저장 경로
  paths: {
    results: './tests/results',
    screenshots: './tests/screenshots',
    logs: './tests/logs'
  },

  // 재시도 설정
  retry: {
    maxAttempts: 3,
    delay: 1000
  }
};
```

---

## 결과 파일 구조

```
tests/
├── coordinator.js              # 메인 조정자
├── shared/
│   ├── config.js               # 설정 (포트, 뷰포트 등)
│   ├── logger.js               # 로깅 유틸리티
│   └── context.js              # 에이전트 간 데이터 공유
├── agents/
│   ├── explorer/index.js       # 페이지 탐색
│   ├── functional/index.js     # 기능 테스트
│   ├── visual/index.js         # 시각/접근성 테스트
│   ├── api/index.js            # API/네트워크 테스트
│   └── performance/index.js    # 성능 테스트
├── results/                     # 테스트 결과 (자동 생성)
│   ├── REPORT.md               # 통합 마크다운 리포트
│   ├── page-map.json           # 페이지 구조 맵
│   ├── functional-tests.json   # 기능 테스트 결과
│   ├── visual-tests.json       # 시각 테스트 결과
│   ├── api-tests.json          # API 테스트 결과
│   ├── network-logs.json       # 네트워크 로그
│   ├── performance-report.json # 성능 리포트
│   ├── context.json            # 공유 컨텍스트
│   ├── agent-status.json       # 실시간 상태
│   └── final-report.json       # 최종 JSON 리포트
├── screenshots/                 # 스크린샷 (자동 생성)
└── logs/                        # 에이전트별 로그 (자동 생성)
```

---

## 개별 에이전트 실행

```bash
# 특정 에이전트만 실행
node -e "
const Agent = require('./tests/agents/explorer');  // 또는 functional, visual, api, performance
const {chromium} = require('playwright');
(async () => {
  const browser = await chromium.launch({ headless: false });
  const context = await browser.newContext();
  const agent = new Agent(context);
  await agent.run();
  await browser.close();
})();
"
```

---

## 특정 에이전트 비활성화

`tests/shared/config.js`에서:
```javascript
agents: {
  explorer: { enabled: true, timeout: 15 * 60 * 1000 },
  functional: { enabled: true, timeout: 15 * 60 * 1000 },
  visual: { enabled: false, timeout: 15 * 60 * 1000 },  // 비활성화
  api: { enabled: true, timeout: 15 * 60 * 1000 },
  performance: { enabled: false, timeout: 15 * 60 * 1000 }  // 비활성화
}
```

---

## 완료 시그널

테스트가 성공적으로 완료되면 다음 출력이 표시됩니다:
```
<promise>MULTI_AGENT_COMPLETE</promise>
```

자동화 스크립트에서 이 문자열을 감지하여 완료 여부 확인 가능

---

## 에러 처리

- 개별 에이전트 실패 시 다른 에이전트는 계속 실행
- 각 에이전트 최대 15분 타임아웃 (config에서 조정 가능)
- 실시간 상태: `tests/results/agent-status.json`
- 에러 목록: `tests/results/context.json`의 `errors` 배열

---

## 트러블슈팅

### 브라우저가 설치되지 않음
```bash
npx playwright install chromium
```

### 포트가 다름
```bash
TEST_URL=http://localhost:YOUR_PORT npm test
```

### 타임아웃 발생
`tests/shared/config.js`에서 타임아웃 증가:
```javascript
agents: {
  explorer: { enabled: true, timeout: 30 * 60 * 1000 },  // 30분
  // ...
}
```

### 특정 페이지 제외
`tests/agents/explorer/index.js`의 `explorePage()` 시작 부분에 추가:
```javascript
// 제외할 경로
if (urlPath.includes('/admin') || urlPath.includes('/internal')) return;
```

---

## CI/CD 통합

### GitHub Actions 예시
```yaml
- name: Run Multi-Agent Tests
  run: |
    npm ci
    npx playwright install chromium
    npm run dev &
    sleep 10
    npm test
  env:
    TEST_URL: http://localhost:3000
    HEADLESS: true
```

### 결과 아티팩트 저장
```yaml
- name: Upload Test Results
  uses: actions/upload-artifact@v3
  with:
    name: test-results
    path: |
      tests/results/
      tests/screenshots/
```

---

## 주의사항

1. **서버 필수**: 테스트 전 대상 서버가 실행 중이어야 합니다
2. **포트 확인**: `TEST_URL` 환경 변수 또는 `config.js`에서 올바른 포트 설정
3. **Chromium 필요**: 첫 실행 시 `npx playwright install chromium` 필요
4. **결과 덮어쓰기**: 매 실행 시 이전 결과가 덮어씌워짐 (백업 필요시 수동 복사)
